<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Grn;
use App\Models\GrnItem;
use App\Models\Product;
use App\Models\ProductPackSize;
use App\Models\Stock;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class InventoryController extends Controller
{
    public function index()
    {
        $stocks = Stock::with('product')->get();
        return view('admin.inventory.index', compact('stocks'));
    }

    public function grnIndex()
    {
        $grns = Grn::with(['creator', 'items'])->latest()->paginate(10);
        return view('admin.inventory.grn.index', compact('grns'));
    }

    public function grnCreate()
    {
        $products = Product::with('packSizes')->where('is_active', true)->get();
        return view('admin.inventory.grn.create', compact('products'));
    }

    public function grnStore(Request $request)
    {
        $request->validate([
            'supplier_name' => 'required|string',
            'received_date' => 'required|date',
            'items' => 'required|array',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.product_pack_size_id' => 'required|exists:product_pack_sizes,id',
            // Simplified validation for now
        ]);

        DB::transaction(function () use ($request) {
            $grn = Grn::create([
                'grn_number' => 'GRN-' . time(), // Simple auto-gen
                'supplier_name' => $request->supplier_name,
                'received_date' => $request->received_date,
                'notes' => $request->notes,
                'created_by' => Auth::id(),
                'total_amount' => 0, // Calculated below
            ]);

            $totalAmount = 0;

            foreach ($request->items as $item) {
                // Determine quantity in base units
                $packSize = ProductPackSize::find($item['product_pack_size_id']);
                $quantityReceived = $item['quantity'];
                $baseQuantity = $quantityReceived * $packSize->quantity;

                $totalCost = $item['cost_price'] * $quantityReceived;
                $totalAmount += $totalCost;

                GrnItem::create([
                    'grn_id' => $grn->id,
                    'product_id' => $item['product_id'],
                    'product_pack_size_id' => $item['product_pack_size_id'],
                    'quantity' => $quantityReceived,
                    'cost_price' => $item['cost_price'],
                    'total_cost' => $totalCost,
                ]);

                // Update Stock
                $stock = Stock::firstOrCreate(
                    ['product_id' => $item['product_id']],
                    ['quantity' => 0]
                );
                $stock->increment('quantity', $baseQuantity);
            }

            $grn->update(['total_amount' => $totalAmount]);
        });

        return redirect()->route('admin.inventory.grn.index')->with('success', 'GRN Created and Stock Updated!');
    }
}
