<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Commission;
use App\Models\Payroll;
use App\Models\SalaryAdvance;
use App\Models\Loan;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class PayrollController extends Controller
{
    public function index(Request $request)
    {
        $month = $request->input('month', now()->format('Y-m'));
        
        $startOfMonth = Carbon::parse($month)->startOfMonth();
        $endOfMonth = Carbon::parse($month)->endOfMonth();

        $reps = User::whereHas('role', function($q) {
            $q->where('name', 'Sales Representative');
        })->with(['payrolls' => function($q) use ($month) {
            $q->where('month', $month);
        }])->get();

        $payrollData = $reps->map(function($rep) use ($startOfMonth, $endOfMonth, $month) {
            // Get total eligible commission (approved)
            $totalCommission = Commission::where('sales_rep_id', $rep->id)
                ->whereBetween('created_at', [$startOfMonth, $endOfMonth])
                ->where('status', 'approved')
                ->sum('commission_amount');

            // Calculate gross salary
            $basicSalary = $rep->basic_salary ?? 0;
            $grossSalary = $basicSalary + $totalCommission;

            // Calculate EPF/ETF
            $epfEmployee = round($grossSalary * 0.08, 2); // 8%
            $etfEmployee = round($grossSalary * 0.03, 2); // 3%
            $epfEmployer = round($grossSalary * 0.12, 2); // 12%
            $etfEmployer = round($grossSalary * 0.03, 2); // 3%

            // Get pending salary advances
            $salaryAdvanceDeduction = SalaryAdvance::where('user_id', $rep->id)
                ->whereIn('status', ['pending', 'partially_deducted'])
                ->sum('remaining_amount');

            // Get active loan installment
            $loanDeduction = Loan::where('user_id', $rep->id)
                ->where('status', 'active')
                ->where('start_deduction_date', '<=', $endOfMonth)
                ->sum('monthly_installment');

            // Total deductions
            $totalDeductions = $epfEmployee + $etfEmployee + $salaryAdvanceDeduction + $loanDeduction;

            // Net salary
            $netSalary = $grossSalary - $totalDeductions;

            // Check if payroll already exists
            $existingPayroll = $rep->payrolls->first();

            return [
                'rep' => $rep,
                'basic_salary' => $basicSalary,
                'total_commission' => $totalCommission,
                'gross_salary' => $grossSalary,
                'epf_employee' => $epfEmployee,
                'etf_employee' => $etfEmployee,
                'epf_employer' => $epfEmployer,
                'etf_employer' => $etfEmployer,
                'salary_advance_deduction' => $salaryAdvanceDeduction,
                'loan_deduction' => $loanDeduction,
                'total_deductions' => $totalDeductions,
                'net_salary' => $netSalary,
                'payroll' => $existingPayroll,
            ];
        });

        return view('admin.payroll.index', compact('payrollData', 'month'));
    }

    public function generate(Request $request)
    {
        $month = $request->input('month', now()->format('Y-m'));
        $paymentDate = $request->input('payment_date', now()->format('Y-m-d'));
        
        $startOfMonth = Carbon::parse($month)->startOfMonth();
        $endOfMonth = Carbon::parse($month)->endOfMonth();

        $reps = User::whereHas('role', function($q) {
            $q->where('name', 'Sales Representative');
        })->get();

        DB::beginTransaction();
        try {
            foreach ($reps as $rep) {
                // Check if payroll already exists
                $existingPayroll = Payroll::where('user_id', $rep->id)
                    ->where('month', $month)
                    ->first();

                if ($existingPayroll) {
                    continue; // Skip if already generated
                }

                // Get total eligible commission
                $totalCommission = Commission::where('sales_rep_id', $rep->id)
                    ->whereBetween('created_at', [$startOfMonth, $endOfMonth])
                    ->where('status', 'approved')
                    ->sum('commission_amount');

                // Calculate gross salary
                $basicSalary = $rep->basic_salary ?? 0;
                $grossSalary = $basicSalary + $totalCommission;

                // Calculate EPF/ETF
                $epfEmployee = round($grossSalary * 0.08, 2);
                $etfEmployee = round($grossSalary * 0.03, 2);
                $epfEmployer = round($grossSalary * 0.12, 2);
                $etfEmployer = round($grossSalary * 0.03, 2);

                // Get pending salary advances
                $salaryAdvances = SalaryAdvance::where('user_id', $rep->id)
                    ->whereIn('status', ['pending', 'partially_deducted'])
                    ->get();
                
                $salaryAdvanceDeduction = 0;
                foreach ($salaryAdvances as $advance) {
                    $deductAmount = min($advance->remaining_amount, $grossSalary * 0.5); // Max 50% of gross
                    $salaryAdvanceDeduction += $deductAmount;
                    
                    // Update advance
                    $advance->deducted_amount += $deductAmount;
                    $advance->remaining_amount -= $deductAmount;
                    if ($advance->remaining_amount <= 0) {
                        $advance->status = 'fully_deducted';
                    } else {
                        $advance->status = 'partially_deducted';
                    }
                    $advance->save();
                }

                // Get active loans
                $loans = Loan::where('user_id', $rep->id)
                    ->where('status', 'active')
                    ->where('start_deduction_date', '<=', $endOfMonth)
                    ->get();
                
                $loanDeduction = 0;
                foreach ($loans as $loan) {
                    if ($loan->paid_installments < $loan->total_installments) {
                        $loanDeduction += $loan->monthly_installment;
                        
                        // Update loan
                        $loan->paid_installments += 1;
                        $loan->paid_amount += $loan->monthly_installment;
                        $loan->remaining_amount -= $loan->monthly_installment;
                        
                        if ($loan->paid_installments >= $loan->total_installments) {
                            $loan->status = 'completed';
                        }
                        $loan->save();
                    }
                }

                // Total deductions
                $totalDeductions = $epfEmployee + $etfEmployee + $salaryAdvanceDeduction + $loanDeduction;

                // Net salary
                $netSalary = $grossSalary - $totalDeductions;

                // Create payroll record
                Payroll::create([
                    'user_id' => $rep->id,
                    'month' => $month,
                    'payment_date' => $paymentDate,
                    'basic_salary' => $basicSalary,
                    'total_commission' => $totalCommission,
                    'gross_salary' => $grossSalary,
                    'epf_employee' => $epfEmployee,
                    'etf_employee' => $etfEmployee,
                    'epf_employer' => $epfEmployer,
                    'etf_employer' => $etfEmployer,
                    'salary_advance_deduction' => $salaryAdvanceDeduction,
                    'loan_deduction' => $loanDeduction,
                    'total_deductions' => $totalDeductions,
                    'net_salary' => $netSalary,
                    'status' => 'processed',
                ]);
            }

            DB::commit();
            return back()->with('success', 'Payroll generated successfully for ' . $month);
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Error generating payroll: ' . $e->getMessage());
        }
    }

    public function show($id)
    {
        $payroll = Payroll::with('user')->findOrFail($id);
        return view('admin.payroll.show', compact('payroll'));
    }

    public function markAsPaid($id)
    {
        $payroll = Payroll::findOrFail($id);
        $payroll->status = 'paid';
        $payroll->save();

        return back()->with('success', 'Payroll marked as paid');
    }

    public function updateBasicSalary(Request $request, $id)
    {
        $request->validate(['basic_salary' => 'required|numeric|min:0']);
        $user = User::findOrFail($id);
        $user->basic_salary = $request->basic_salary;
        $user->save();
        
        return back()->with('success', 'Basic Salary updated.');
    }

    public function updateEmployeeDetails(Request $request, $id)
    {
        $request->validate([
            'epf_number' => 'nullable|string|max:255',
            'etf_number' => 'nullable|string|max:255',
            'bank_name' => 'nullable|string|max:255',
            'bank_account_number' => 'nullable|string|max:255',
            'bank_branch' => 'nullable|string|max:255',
        ]);

        $user = User::findOrFail($id);
        $user->update($request->only([
            'epf_number',
            'etf_number',
            'bank_name',
            'bank_account_number',
            'bank_branch'
        ]));

        return back()->with('success', 'Employee details updated successfully.');
    }

    public function recalculate($id)
    {
        $payroll = Payroll::findOrFail($id);
        
        if ($payroll->status === 'paid') {
            return back()->with('error', 'Cannot recalculate a paid payroll.');
        }

        DB::beginTransaction();
        try {
            $user = User::findOrFail($payroll->user_id);
            $month = $payroll->month;
            $startOfMonth = Carbon::parse($month)->startOfMonth();
            $endOfMonth = Carbon::parse($month)->endOfMonth();

            // 1. Re-fetch Commission
            $totalCommission = Commission::where('sales_rep_id', $user->id)
                ->whereBetween('created_at', [$startOfMonth, $endOfMonth])
                ->where('status', 'approved')
                ->sum('commission_amount');

            // 2. Base & Gross
            $basicSalary = $user->basic_salary ?? 0;
            $grossSalary = $basicSalary + $totalCommission;

            // 3. Deductions
            $epfEmployee = round($grossSalary * 0.08, 2);
            $etfEmployee = round($grossSalary * 0.03, 2); // 3%
            $epfEmployer = round($grossSalary * 0.12, 2);
            $etfEmployer = round($grossSalary * 0.03, 2);

            // Keep existing advances/loans deductions
            $salaryAdvanceDeduction = $payroll->salary_advance_deduction; 
            $loanDeduction = $payroll->loan_deduction;
            
            $totalDeductions = $epfEmployee + $etfEmployee + $salaryAdvanceDeduction + $loanDeduction;
            $netSalary = $grossSalary - $totalDeductions;

            $payroll->update([
                'basic_salary' => $basicSalary,
                'total_commission' => $totalCommission,
                'gross_salary' => $grossSalary,
                'epf_employee' => $epfEmployee,
                'etf_employee' => $etfEmployee,
                'epf_employer' => $epfEmployer,
                'etf_employer' => $etfEmployer,
                'total_deductions' => $totalDeductions,
                'net_salary' => $netSalary,
            ]);

            DB::commit();
            return back()->with('success', 'Payroll recalculated successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Error recalculating: ' . $e->getMessage());
        }
    }
}
