<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\ProductPackSize;
use App\Models\ProductPriceLevel;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

class ProductController extends Controller
{
    public function index()
    {
        $products = Product::with('packSizes')->latest()->paginate(10);
        return view('admin.products.index', compact('products'));
    }

    public function create()
    {
        return view('admin.products.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'base_cost' => 'required|numeric|min:0',
            'image' => 'nullable|image|max:2048', // 2MB Max
            
            // Cartoon Config
            'carton_length' => 'required|numeric|min:0',
            'carton_width' => 'required|numeric|min:0',
            'carton_height' => 'required|numeric|min:0',
            'carton_weight' => 'required|numeric|min:0',
            'pcs_per_carton' => 'required|integer|min:1',

            // Prices for Unit (Base Pack)
            'price_vvip' => 'required|numeric|min:0',
            'price_vip' => 'required|numeric|min:0',
            'price_wholesale' => 'required|numeric|min:0',
            'price_retail' => 'required|numeric|min:0',
        ]);

        DB::transaction(function () use ($request) {
            // 1. Create Product
            $imagePath = null;
            if ($request->hasFile('image')) {
                $imagePath = $request->file('image')->store('products', 'public');
            }

            $product = Product::create([
                'name' => $request->name,
                'base_cost' => $request->base_cost,
                'image' => $imagePath,
                'is_active' => true,
            ]);

            // 2. Create "Unit" Pack Size
            $unitPack = ProductPackSize::create([
                'product_id' => $product->id,
                'size_name' => 'Unit',
                'quantity' => 1,
            ]);

            // 3. Create Price Levels for Unit
            $levels = [
                'VVIP' => $request->price_vvip,
                'VIP' => $request->price_vip,
                'Wholesale' => $request->price_wholesale,
                'Retail' => $request->price_retail,
            ];

            foreach ($levels as $levelName => $price) {
                ProductPriceLevel::create([
                    'product_pack_size_id' => $unitPack->id,
                    'price_level' => $levelName,
                    'price' => $price,
                ]);
            }

            // 4. Create "Carton" Pack Size
            $cartonPack = ProductPackSize::create([
                'product_id' => $product->id,
                'size_name' => 'Carton',
                'quantity' => $request->pcs_per_carton,
                'length' => $request->carton_length,
                'width' => $request->carton_width,
                'height' => $request->carton_height,
                'weight' => $request->carton_weight,
            ]);

            // For now, we only asked regarding UNIT prices, but usually Cartons also have prices. 
            // I'll auto-calculate or leave empty? The prompt says "4 price levels for one unit".
            // I will assume for now Carton prices are not manually set or are calculated xQty.
            // But to be safe, I'll add the same levels x Qty as a default, or just leave it. 
            // Better to add them so the system works:
            foreach ($levels as $levelName => $price) {
                ProductPriceLevel::create([
                    'product_pack_size_id' => $cartonPack->id,
                    'price_level' => $levelName,
                    'price' => $price * $request->pcs_per_carton, // Auto-calc for now
                ]);
            }

        });

        return redirect()->route('admin.products.index')->with('success', 'Product Created Successfully!');
    }
}
