<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\ProductPackSize;
use App\Models\Stock;
use App\Models\Vehicle;
use App\Models\VehicleStock; // Need to create Model
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;

class StockTransferController extends Controller
{
    public function create()
    {
        $vehicles = Vehicle::where('is_active', true)->get();
        // Get products with stocks > 0
        $products = Product::whereHas('packSizes')->get(); 
        
        // Get available Sales Reps
        $salesReps = \App\Models\User::whereHas('role', function($q) {
            $q->where('name', 'Sales Representative');
        })->where('is_active', true)->get();
        
        return view('admin.stock_transfer.create', compact('vehicles', 'products', 'salesReps'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'vehicle_id' => 'required|exists:vehicles,id',
            'sales_rep_id' => 'required|exists:users,id', // Validate Rep Selection
            'items' => 'required|array',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.product_pack_size_id' => 'required|exists:product_pack_sizes,id',
            'items.*.quantity' => 'required|integer|min:1',
        ]);

        DB::transaction(function () use ($request) {
            
            // 0. Assign Sales Rep to Vehicle
            $rep = \App\Models\User::find($request->sales_rep_id);
            $rep->vehicle_id = $request->vehicle_id;
            $rep->save();

            foreach ($request->items as $item) {
                $packSize = ProductPackSize::find($item['product_pack_size_id']);
                $totalBaseUnits = $item['quantity'] * $packSize->quantity;

                // 1. Check Warehouse Stock
                $warehouseStock = Stock::where('product_id', $item['product_id'])->first();
                
                if (!$warehouseStock || $warehouseStock->quantity < $totalBaseUnits) {
                    throw new \Exception("Insufficient stock for product ID: " . $item['product_id']);
                }

                // 2. Deduct from Warehouse
                $warehouseStock->decrement('quantity', $totalBaseUnits);

                // 3. Add to Vehicle Stock
                $vehicleStock = VehicleStock::firstOrCreate(
                    ['vehicle_id' => $request->vehicle_id, 'product_id' => $item['product_id']],
                    ['quantity' => 0]
                );
                $vehicleStock->increment('quantity', $totalBaseUnits);

                // 4. Log Transfer (Optional but good)
                // Assuming StockTransfer model exists or created dynamically
                \DB::table('stock_transfers')->insert([
                    'transfer_date' => now(),
                    'vehicle_id' => $request->vehicle_id,
                    'product_id' => $item['product_id'],
                    'product_pack_size_id' => $item['product_pack_size_id'],
                    'pack_quantity' => $item['quantity'],
                    'converted_base_quantity' => $totalBaseUnits,
                    'created_by' => Auth::id(),
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
            }
        });

        return redirect()->route('admin.dashboard')->with('success', 'Stock loaded to truck and Sales Rep assigned successfully!');
    }
}
