<?php

namespace App\Http\Controllers\Sales;

use App\Http\Controllers\Controller;
use App\Models\Customer;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class CustomerController extends Controller
{
    public function index()
    {
        $customers = Customer::where('sales_rep_id', Auth::id())
            ->with(['sales' => function($query) {
                // We'll use this to calculate outstanding, though total_due might be a direct attribute or calculation
            }])
            ->get();
            
        // Assuming total_due is an accessor or needs calculation
        foreach($customers as $customer) {
            // Recalculate or ensure total_due is available
            // If it's not a column, we formulate it:
            $customer->outstanding = $customer->sales->sum('balance_due');
        }

        return view('sales.customers.index', compact('customers'));
    }

    public function create()
    {
        return view('sales.customers.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'mobile' => 'required|string|max:20|unique:customers,mobile',
            'address' => 'required|string|max:500',
            'area' => 'nullable|string|max:255',
            'gst_number' => 'nullable|string|max:50',
            'default_price_level' => 'required|in:VVIP,VIP,Wholesale,Retail',
        ]);

        Customer::create([
            'name' => $request->name,
            'mobile' => $request->mobile,
            'address' => $request->address,
            'area' => $request->area,
            'gst_number' => $request->gst_number,
            'default_price_level' => $request->default_price_level,
            'sales_rep_id' => Auth::id(),
            'is_active' => true,
        ]);

        return redirect()->route('sales.customers.index')->with('success', 'Customer created successfully!');
    }

    public function show(Customer $customer)
    {
        if ($customer->sales_rep_id !== Auth::id()) {
            abort(403, 'Unauthorized action.');
        }

        $invoices = $customer->sales()->latest()->get();
        
        // Group payments by receipt_number
        $receipts = $customer->payments()
            ->get()
            ->groupBy('receipt_number')
            ->map(function ($group) {
                return (object) [
                    'receipt_number' => $group->first()->receipt_number,
                    'date' => $group->first()->created_at,
                    'amount' => $group->sum('amount'),
                    'ids' => $group->pluck('id')->implode(',')
                ];
            })
            ->sortByDesc('date');

        return view('sales.customers.show', compact('customer', 'invoices', 'receipts'));
    }
}
