<?php

namespace Database\Seeders;

use App\Models\User;
use App\Models\Role;
use App\Models\Customer;
use App\Models\Product;
use App\Models\ProductPackSize;
use App\Models\ProductPriceLevel;
use App\Models\Vehicle;
// use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Hash;

class DatabaseSeeder extends Seeder
{
    /**
     * Seed the application's database.
     */
    public function run(): void
    {
        // 1. Create Roles
        $adminRole = Role::firstOrCreate(['name' => 'Admin']);
        $repRole = Role::firstOrCreate(['name' => 'Sales Representative']);

        // 2. Create Admin User
        User::firstOrCreate(
            ['email' => 'admin@example.com'],
            [
                'name' => 'Admin User',
                'password' => Hash::make('password'),
                'role_id' => $adminRole->id,
                'is_active' => true,
            ]
        );

        // 3. Create Rep User
        $repUser = User::firstOrCreate(
            ['email' => 'rep1@example.com'],
            [
                'name' => 'Sales Rep 1',
                'password' => Hash::make('password'),
                'role_id' => $repRole->id,
                'mobile' => '0712345678',
                'is_active' => true,
            ]
        );

        // 4. Create Product with Packs and Prices
        $product1 = Product::firstOrCreate(['name' => 'Marie Biscuits 400g']);
        
        // 54: Packs
        $packUnit = ProductPackSize::firstOrCreate(
            ['product_id' => $product1->id, 'size_name' => 'Unit'],
            ['quantity' => 1]
        );
        
        $packCarton = ProductPackSize::firstOrCreate(
            ['product_id' => $product1->id, 'size_name' => 'Carton (24x)'],
            ['quantity' => 24]
        );

        // 66: Prices for Unit
        ProductPriceLevel::updateOrCreate(
            ['product_pack_size_id' => $packUnit->id, 'price_level' => 'retail'],
            ['price' => 150.00]
        );
        ProductPriceLevel::updateOrCreate(
            ['product_pack_size_id' => $packUnit->id, 'price_level' => 'wholesale'],
            ['price' => 135.00]
        );
        
        // 70: Prices for Carton
        ProductPriceLevel::updateOrCreate(
            ['product_pack_size_id' => $packCarton->id, 'price_level' => 'retail'],
            ['price' => 3600.00]
        );
        ProductPriceLevel::updateOrCreate(
            ['product_pack_size_id' => $packCarton->id, 'price_level' => 'wholesale'],
            ['price' => 3000.00]
        );
        
        // 74: Create Vehicle
        $vehicle = Vehicle::firstOrCreate(
            ['truck_number' => 'WP-AA-1234'],
            ['driver_name' => 'Suresh Driver']
        );

        // 77: Create Customer
        $customer1 = Customer::firstOrCreate(
            ['mobile' => '0777999888'],
            [
                'name' => 'Colombo Super City',
                'area' => 'Colombo 03',
                'credit_limit' => 100000,
                'sales_rep_id' => $repUser->id,
            ]
        );
        
        $customer2 = Customer::firstOrCreate(
            ['mobile' => '0711555666'],
            [
                'name' => 'Village Store',
                'area' => 'Kandy',
                'credit_limit' => 20000,
                'default_price_level' => 'wholesale',
                'sales_rep_id' => $repUser->id,
            ]
        );
        // 94: Create Sales (Invoice)
        $sale1 = \App\Models\Sale::firstOrCreate(
            ['invoice_number' => 'INV-001'],
            [
                'customer_id' => $customer1->id,
                'sales_rep_id' => $repUser->id,
                'vehicle_id' => $vehicle->id,
                'sub_total' => 300.00,
                'net_total' => 300.00,
                'payment_status' => 'paid',
                'payment_type' => 'Cash',
                'status' => 'completed',
            ]
        );

        \App\Models\SaleItem::firstOrCreate(
            [
                'sale_id' => $sale1->id,
                'product_id' => $product1->id,
                'product_pack_size_id' => $packUnit->id,
            ],
            [
                'price_level_applied' => 'retail',
                'quantity' => 2,
                'unit_price' => 150.00,
                'total_price' => 300.00,
            ]
        );

        // 117: Create Payments
        // Cash Payment
        \App\Models\Payment::firstOrCreate(
            [
                'sale_id' => $sale1->id,
                'method' => 'Cash',
            ],
            [
                'customer_id' => $customer1->id,
                'amount' => 300.00,
                'payment_date' => now(),
                'collected_by' => $repUser->id,
            ]
        );

        // Cheque Payment (generic on account)
        \App\Models\Payment::firstOrCreate(
            [
                'reference' => 'CHQ-998877',
            ],
            [
                'customer_id' => $customer2->id,
                'amount' => 5000.00,
                'method' => 'Cheque',
                'payment_date' => now(),
                'collected_by' => $repUser->id,
            ]
        );
        // 137: Create Stocks (Initial Inventory)
        \App\Models\Stock::updateOrCreate(
            ['product_id' => $product1->id],
            ['quantity' => 5000] // Reset to 5000 if running seeder
        );

        // 143: Create GRN (Goods Received Note)
        $grn = \App\Models\Grn::firstOrCreate(
            ['grn_number' => 'GRN-2026-001'],
            [
                'supplier_name' => 'Biscuits Factory Ltd',
                'received_date' => now()->subDays(5),
                'total_amount' => 120000.00,
                'notes' => 'Initial stock for Jan 2026',
                'created_by' => 1,
            ]
        );
        
        // GRN Items
        \App\Models\GrnItem::firstOrCreate(
            [
                'grn_id' => $grn->id,
                'product_id' => $product1->id,
                'product_pack_size_id' => $packCarton->id,
            ],
            [
                'quantity' => 40, 
                'cost_price' => 3000.00, 
                'total_cost' => 120000.00,
            ]
        );

        // 163: Create Expense
        \App\Models\Expense::create([
            'user_id' => $repUser->id,
            'category' => 'Fuel',
            'amount' => 5000.00,
            'description' => 'Diesel for truck WP-AA-1234',
            'expense_date' => now(),
            'status' => 'pending',
        ]);
    }
}
