<?php

namespace Tests\Feature;

use App\Models\User;
use App\Models\Role;
use App\Models\Customer;
use App\Models\Product;
use App\Models\ProductPackSize;
use App\Models\Vehicle;
use App\Models\VehicleStock;
use App\Models\Sale;
use App\Models\Commission;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\DB;
use Tests\TestCase;

class CommissionCalculationTest extends TestCase
{
    // use RefreshDatabase; // Careful with this on a dev environment with existing data. 
    // Since I don't want to wipe the user's DB, I will create data and maybe clean it up or just use transaction.
    // Actually, feature tests usually use a separate DB or in-memory DB. 
    // Given I am running on the user's actual environment, I should be careful.
    // I will use manual cleanup or just transaction rollback if possible, but Laravel's RefreshDatabase traits wipes migration.
    // I'll avoid RefreshDatabase and just create unique data and assert, then maybe leave it or delete it.

    public function test_split_payment_commission_calculation()
    {
        DB::beginTransaction();

        try {
            // 1. Setup Data
            $role = Role::firstOrCreate(['name' => 'Sales Representative']);
            $vehicle = Vehicle::create(['number' => 'TEST-VAN-001', 'type' => 'Van']);
            
            $rep = User::create([
                'name' => 'John Doe Test',
                'email' => 'john.test@example.com',
                'password' => bcrypt('password'),
                'role_id' => $role->id,
                'vehicle_id' => $vehicle->id,
                'is_active' => true,
                // Commission Rates
                'commission_cash' => 10,   // 10%
                'commission_cheque' => 5,  // 5%
                'commission_credit' => 2,  // 2%
            ]);

            $customer = Customer::create([
                'name' => 'Test Customer',
                'mobile' => '0771234567',
                'address' => 'Test Address',
                'default_price_level' => 'Retail',
                'sales_rep_id' => $rep->id,
            ]);

            $product = Product::create([
                'name' => 'Test Product', 
                'code' => 'TP001', 
                'type' => 'Standard'
            ]);
            
            $packSize = ProductPackSize::create([
                'product_id' => $product->id,
                'size_name' => '1kg',
                'quantity' => 1000, // 1kg
            ]);

            // Add Stock to Vehicle
            VehicleStock::create([
                'vehicle_id' => $vehicle->id,
                'product_id' => $product->id,
                'quantity' => 100000, // plenty
            ]);

            // 2. Simulate Billing Request
            // Scenario: 
            // Total Sale: 1000
            // Cash Paid: 500  -> Commission: 500 * 10% = 50
            // Cheque Paid: 300 -> Commission: 300 * 5% = 15
            // Credit (Balance): 200 -> Commission: 200 * 2% = 4
            // Total Expected Commission: 69

            // Login as Rep
            $this->actingAs($rep);

            $response = $this->post(route('sales.billing.store'), [
                'customer_id' => $customer->id,
                'items' => [
                    [
                        'product_id' => $product->id,
                        'product_pack_size_id' => $packSize->id,
                        'quantity' => 10,
                        'unit_price' => 100, // 10 * 100 = 1000 Total
                        'price_level' => 'Retail',
                    ]
                ],
                'payment_amount' => 1000, // Not strictly used by logic but might be for validation? 
                // Wait, logic uses 'payment_cash' and 'payment_cheque' fields. 
                // 'payment_amount' field validation exists: 'payment_amount' => 'required|numeric|min:0'
                // This field seems to be a generic "amount paying now" field from before split payment logic?
                // Let's check the controller validation again.
                // 'payment_amount' is validated but not used in calculation of cash/cheque. 
                // Wait, if I look at BillingController:
                // $cashAmount = $request->payment_cash ?? 0;
                // $chequeAmount = $request->payment_cheque ?? 0;
                // It does NOT use $request->payment_amount for the split calculation.
                // But it IS required in validation. This might be a bug or legacy field.
                // The JS usually updates a total paid field. I should provide it to pass validation.
                
                'payment_amount' => 800, // 500 + 300
                'payment_method' => 'Cash', // Legacy field? Controller validation says 'required|in:Cash,Cheque,Credit'. 
                // But logic derives actual type from amounts. I'll pass 'Cash' to satisfy validation.
                
                'payment_cash' => 500,
                'payment_cheque' => 300,
                // Resulting Balance Due should be 200.
                'cheque_reference' => 'CHQ123',
            ]);

            // 3. Assertions
            $response->assertRedirect();
            
            $sale = Sale::where('sales_rep_id', $rep->id)->latest()->first();
            $this->assertNotNull($sale, 'Sale was not created');
            $this->assertEquals(1000, $sale->net_total, 'Net total incorrect');
            $this->assertEquals(800, $sale->paid_amount, 'Paid amount incorrect');
            $this->assertEquals(200, $sale->balance_due, 'Balance due incorrect');
            
            $commission = Commission::where('sale_id', $sale->id)->first();
            $this->assertNotNull($commission, 'Commission record not created');
            
            // Expected: 50 (Cash) + 15 (Cheque) + 4 (Credit) = 69
            $this->assertEquals(69, $commission->commission_amount, 'Commission calculation incorrect');
            
            echo "\nTest Passed! Commission: " . $commission->commission_amount . "\n";

        } catch (\Exception $e) {
            echo "\nTest Failed: " . $e->getMessage() . "\n";
            throw $e;
        } finally {
            DB::rollBack(); // Clean up DB so we don't leave trash
        }
    }
}
